; Debugging.asm - demonstrates how to more easily debug your applications
;
; Up to now, we've only had fairly simple programs, the debugging of
; which is quite simple - re-check the code to spot the offending line(s)
; or remove bits until it works and we then find out where the problem
; was. It should be pretty obvious that this isn't going to be very
; effective once we move beyond the simplest of programs. We've got a
; couple of options:
;
;   1. MessageBox's, an extremely crude way of signalling where the
;      program's up to, but sometimes sufficient.
;
;   2. Using a debugger to trace through the program, a useful
;      approach if your debugger and linker can work together to
;      provide symbols in the debugger, though it can be very
;      heavy going for the novice at this.
;
;   3. Using the OutputDebugString API to send strings to a
;      debugger or other program which can capture them. By sending,
;      I mean we pass a string to the function and it sends the
;      whole string to the debugger (or receiving program).
;
; In this example, I'll demonstrate how to put the third option to
; use, without the need for a debugger.
;
; In the \Bin directory, you'll find a nice program called DebugView
; (Dbgview.exe) which can capture messages sent by OutputDebugString
; in another window. With this program, we can output strings via
; the API and have DebugView capture them, thus giving us a way of
; generating a log of debug messages.
;
; Still, just strings is a bit limiting. What would be nice is if
; we could output register values, etc. Fortunately, I've made a
; macro that can handle this:
;
;   debug <string to output>, <items>
;
; The string to output is in the same form as C's printf strings, ie
; it can contain %d format characters, etc. The <items> should be a
; list of the values to replace the format characters with and can
; evaluate to any acceptable assembler expression. Some examples
; should help make this clear:
;
;   debug 'eax=%#08x, ebx=%#08x, ecx=%#08x', eax, ebx, ecx
;   debug 'Coords: x=%d, y=%d', [esi], [esi + 4]
;   debug 'Coords: x=%d, y=%d', [_coords_x], [_coords_y]
;
; Essentially, debug can be thought of as printf except the output
; goes to the debugger (or DebugView). You may want to dig out an
; old C book or search the web for the full formatting capabilities.
;
; Note that all parameters passed are assumed to be dword in size.
;
; For this example, start DebugView and then make sure that the 'Force
; Carriage Returns' option is selected in the 'Options' menu. This sets
; DebugView up to capture our debugging messages.
;
; If you get a missing DLL error, you need to install the TCP/IP
; protocol onto your system and ensure you're running Winsock 2. If
; you're on Windows 95, run W95ws2setup.exe to update your system.
; If it fails, you can try renaming ws2_32_.dll to ws2_32.dll.
;
; Once DebugView is running, start up the example and left/right
; click to see the messages produced. During development, you may
; find it useful to just leave DebugView running.
;

%define _WINMESSAGES_
%define _WINVKEYS_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	WinMainPrologue
	mov	esi, ._wndclass
	mov	edi, ._msg
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	sc RegisterClassEx, esi
	cmp	eax, TRUE
	je	near _WinMain_Fail
	TEXTlocal szWndCaption, 'Debugging example',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	mov	._hwnd, eax
	sc ShowWindow, ._hwnd, .nCmdShow
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	sc GetMessage, ._msg, NULL, 0, 0
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	sc TranslateMessage, ._msg
	sc DispatchMessage, ._msg
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
proc _WndProc, hwnd, message, wParam, lParam
	ddlocal		_hdc
	struclocal	_rect, RECT, _ps, PAINTSTRUCT
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_PAINT
			sc BeginPaint, .hwnd, ._ps
			mov	._hdc, eax
			sc GetClientRect, .hwnd, ._rect
			TEXTlocal _lpOutputString, 'Click the left/right mouse buttons',0
			sc DrawText, ._hdc, ._lpOutputString, -1, ._rect, DT_SINGLELINE | DT_CENTER | DT_VCENTER
			sc EndPaint, .hwnd, ._ps
			xor	eax, eax
			break
		case WM_LBUTTONDOWN
			debug 'Left button down - register dump:'
			debug 'eax=%#08x, ebx=%#08x, ecx=%#08x, edx=%#08x', eax, ebx, ecx, edx
			debug 'esi=%#08x, edi=%#08x, ebp=%#08x', esi, edi, ebp
			xor	eax, eax
			break
		case WM_LBUTTONUP
			debug 'Left button up'
			debug '-'
			xor	eax, eax
			break
		case WM_RBUTTONDOWN
			debug 'Right button down - memory addresses:'
			debug '_temp_addr1=%#08x, _temp_addr2=%#08x', _temp_addr1, _temp_addr2
			xor	eax, eax
			break
		case WM_RBUTTONUP
			debug 'Right button up'
			debug '-'
			xor	eax, eax
			break
		case WM_DESTROY
			sc PostQuitMessage, 0
			xor	eax, eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc

[section .bss]
_temp_addr1	resd 1
_temp_addr2	resd 1

[section .data]
